function [dairyMS, sheepSR ] = LUIM2_land_intensity_calculation(GISINTENSITY, c, LU_endogenous, LU_exogenous, dairy_coef, dairy_scaling_coef, sheep_scaling_coef, last_year, ALL_YEARS )
%
% Land Use Intensity Module - function 2 (LUIM2)
%
% This file calculates land use intensity spatially.
% For dairy farming this is milk solds per hectare.
% For sheep/beef farming this is stock units per hectare.
%
%   13-Jan-12 : Dairy and Sheep/beef only
%

% Code by Simon Anastasiadis : 2012-01-13

%% Parameters

% base year
base_year = 2008;

%% Identify years of interest

% construct vector containing years of interest
if ALL_YEARS
    % we want all years
    years = (base_year:last_year).';
else
    % we want just the final year
    years = last_year;
end

%% Construct matrix of land uses for years of interest

if ALL_YEARS
    % combine all years
    LU_exogenous = [LU_exogenous(:,1), LU_exogenous(:,2) * ones(1,length(years))];
    % merge
    LU_matrix = [LU_endogenous ; LU_exogenous];
else
    % only keep the last year
    LU_endogenous = LU_endogenous(:,[1, end]);
    % merge
    LU_matrix = [LU_endogenous ; LU_exogenous];
end

% sort rows
LU_matrix = sortrows(LU_matrix,1);
GISINTENSITY = sortrows(GISINTENSITY,1);

% assert LU_matrix sorters are consistent with GISINTENTITY
% 1) assert same number of rows in each matrix
msg = sprintf(['Error in LUIM2\n'...
               'Vectors are different lengths']);
assert( size(LU_matrix,1) == size(GISINTENSITY,1), msg);
% 2) assert rows in each matrix correspond to each other
msg = sprintf(['Error in LUIM2\n'...
               'Sorters do not correspond']);
assert( all(LU_matrix(:,1) == GISINTENSITY(:,1)), msg);

%% Setup storage for results

% dairy
dairyMS = zeros(size(LU_matrix,1),length(years)+1);
% sheep/beef
sheepSR = zeros(size(LU_matrix,1),length(years)+1);

% fill column 1 with sorters
dairyMS(:,1) = GISINTENSITY(:,c.sorter);
sheepSR(:,1) = GISINTENSITY(:,c.sorter);

%% Iterate over the years of interest

for jj = 1:length(years)
    % simulation year
    year = years(jj);
    
    %% Calculate dairy intensity

    % dairy_coef columns are as follows:
    % 1. LIC code
    % 2. alpha
    % 3. beta
    % 4. gamma
    % 5. constant
    % Farms with LIC code have intensity alpha + beta * Ln(year - gamma)
    % or constant (provided coefficients).

    % dairy indicator
    iDairy = LU_matrix(:,jj+1) == 1;

    % loop through dairy regions
    for ii = 1:size(dairy_coef,1)
        % identify LIC region
        iLIC = GISINTENSITY(:,c.lic) == dairy_coef(ii,1);
        % identify pixels of interest
        iLICdairy = iLIC & iDairy;

        % if a constant is provided, use this
        if ~isnan(dairy_coef(ii,5))
            MSha = dairy_coef(ii,5);
        % else calculate by formula
        else
            % calculate
            alpha = dairy_coef(ii,2);
            beta =  dairy_coef(ii,3);
            gamma = dairy_coef(ii,4);
            MSha = alpha + beta * log(year - gamma);
        end

        % assign
        dairyMS(iLICdairy,jj+1) = MSha;
        
        % scale for effective land area
        dairyMS(iLICdairy,jj+1) = dairyMS(iLICdairy,jj+1) * dairy_scaling_coef;

    end

    %% Calculate sheep beef intensity

    % sheep_coef columns are as follows:
    % 1. SB class code
    % 2. alpha
    % 3. beta
    % 4. gamma
    % 5. constant
    % Farms with class code have intensity alpha + beta * Ln(year - gamma)
    % or constant (provided coefficients).

    % sheep beef indicator OR public pasture
    iSheep = LU_matrix(:,jj+1) == 2 | LU_matrix(:,jj+1) == 10;

    % assign sheep/beef stock numbers
    sheepSR(iSheep,jj+1) = GISINTENSITY(iSheep,c.ccav);
    
    % Scale Sheep and Beef numbers by scaling constant
    sheepSR(:,jj+1) = sheepSR(:,jj+1) * sheep_scaling_coef;
    
end

%% Inform User projecting land use change is complete

msg = sprintf(' - - LUIM2 : Land Intensity Calculation complete');
disp(msg)

end



